"""Automate batch simulation for any number of doe2 input & weather files.

https://unmethours.com/question/21471/equest-batch-simulation-runs/
Answered Dec 2 '16
By Nick Caton

Note from site terms of service:
"User Intellectual Property
You agree that all original user contributions that you make to the Site are your intellectual property owned by you and are automatically licensed to Big Ladder Software LLC under the Creative Commons Attribution Share Alike 3.0 License. You grant Big Ladder Software LLC the right to copy, reproduce, display, publish, and redistribute your original user contributions to the Site, and allow others to do so also, under the terms of the license."
https://proxy.unmethours.com/legal.html

Prerequisites:
* python
* tabulate (eg. conda install tabulate)

"""

import os
import glob
import subprocess
import shutil
import re
import sys

"""
DOE command line usage: as distributed with MASControl3
C:/DEER2020/MASControl3DOE2/exe/DOE22R/RunDoe2.bat
(also, for non-refrigeration models, C:/DEER2020/MASControl3/DOE2/exe/DOE23/RunDoe2.bat)
REM %1 is weather file (eg. CZ01)
REM %2 is the INP file root (eg. Gro-CZ01-1975)
REM %3 is SimFiles PATH (eg. C:/DEER2020/MASControl3/InpFiles)
REM %4 is Flag for file save option (ALL causes save BDL and SIM; anything else does not)
cmdargs = [doe2bat_location, weather, inp, simfiles, flag]

Runs sim and saves these files:
%2.LRP
%2.SRP
%2.SIN
%2-SIM.LOG
%2-HRLY.TXT
%2.BDL (if %4 = ALL)
%2.SIM (if %4 = ALL)
"""

# Review these environment variables for running doe2 on command line
# store current working directory
start_dir = os.getcwd() # C:/DEER2020/MASControl3/batch
doe2bat_location = os.path.abspath(os.path.join(start_dir,'../DOE2/exe/DOE23/RunDoe2.bat'))
doe2bat_dir = os.path.dirname(doe2bat_location)
print(doe2bat_location)
if not (os.path.isfile(doe2bat_location)):
    raise FileNotFoundError(doe2bat_location)

def weatherbatch2(w,i,simfiles=".",flag="ALL"):
    """This function will simulate ONE model specified by a file pair of the form
        i = input file (*.inp)
        w = weather file (*.bin)
    The files should be co-located at target_path. The function returns
        output_message = the text output from the command line
    Options:
        append_weather_to_name (default=True):
            Runs the inp file as <inputname>_in_<weathername>.inp,
            useful if you run the same inp file with multiple weather files.
    
    Do your own globbing elsewhere and call this function once for each simulation.
    The simulation is performed via doe2 (the simulation engine behind eQuest), using DOE2
    commandline scripts. 

    The argument 'target_path' is the directory containing the weather (*.bin) 
    and input (*.inp) files.

    When finished, BDL files and SIM output files with self-descriptive names 
    to indicate the associated weather / input combination should remain 
    alongside the targeted files.

    Be mindful limits to doe2 command line execution apply to this script as 
    well.  Excessive characters and the presence of any spaces in the input or
    weather file names will result in errors.
    """

    # strip '.INP' and '.BIN' from w for commandline execution 
    # (done via regex to make this case-insensitive)
    weather = re.sub('\.bin', '', w, flags=re.IGNORECASE)
    # (done via regex to make this case-insensitive)
    inp = re.sub('\.inp', '', i, flags=re.IGNORECASE)
        
    # execute commandline doe2 for this weather + input combination
    cmdargs = [doe2bat_location, weather, inp, simfiles, flag]
    print(cmdargs)
    #output_message = subprocess.check_output(cmdargs)
    subprocess.run(cmdargs,stdout=sys.stdout)
    output_message = b"Done... check command line output.\n"

    return output_message
    
def usage():
    print(
"""Run a bunch of DOE2 simulations with specified weather files.
Usage:
    python doe_batch_process.py <target> [<listfile>]
where
    <target> is the path to your folder of INP [and optionally BIN] files.
    
    <listfile> (optional) is a text file that lists pairs of INP and BIN files, eg.
        inp1.inp weather1.bin
        inp2.inp weather2.bin
    Without this argument, the script will look for "CZ##" in the name of your INP
    file and use that as the weather file.        

For example if your files are in a folder called v2011-and-before, enter
    python doe_batch_process.py v2011-and-before

* Note that you need to have DOE2 set up first (confirm the paths in the script).
* If the weather file is not already installed in doe22/weather, this script will
  copy it there from <target>.
* For more options, you can import this file as a library and run within python.

For help contact Nicholas <nfette@solaris-technical.com>
""")


# What to do if this script is run on command line (instead of imported)
def main():
    import sys
    from timeit import default_timer as timer
    append_weather_to_name = False
    
    if len(sys.argv) > 1:
        usage()
        return

    print("Starting script in ",os.getcwd())
    #establish the target directory
    #target_path = sys.argv[1]
    target_path = os.path.abspath(os.getcwd())+'\\'
    
    
    logfilename="batch_outputs.log"
    with open(logfilename, "at") as logfile:
        import datetime
        print(datetime.datetime.now(),file=logfile)
    
        if len(sys.argv) > 2:
            # Read the listfile
            append_weather_to_name = True
            listfile = sys.argv[2]
            with open(listfile,'rt') as f:
                pairs = [tuple(line.strip().split()) for line in f]
        else:
            # Find the weather file assuming it is CZ## in the inp file name.
            append_weather_to_name = False
            inpfiles = [os.path.basename(i) for i in glob.glob("*.inp")]
            weathers = [re.search('CZ..',inp).group()+".bin" for inp in inpfiles]
            pairs = list(zip(inpfiles, weathers))

        print("Batch processing the following files:",file=logfile)
        print(file=logfile)
        try:
            import tabulate
            print(tabulate.tabulate(pairs, headers=("Inp simulation files","Bin weather files")),
                file=logfile)
        except ModuleNotFoundError:
            print("(Try 'conda install tabulate' for pretty output here)",file=logfile)
            import pprint
            pprint.pprint(pairs,stream=logfile)
        print(file=logfile)

        if False:
            # Fix the path to MASControl inside each INP file.
            # Note that python won't let you end the string with a backslash,
            # so add some extra character like space and slice it off, ie. r"C:\ "[:-1]
            text_to_search=r"##fileprefix C:\MASControl3_01\MASControl3\ "[:-1]
            replacement_text=r"##fileprefix C:\DEER2020\MASControl3\ "[:-1]
            # N.B.: https://stackoverflow.com/questions/647769/why-cant-pythons-raw-string-literals-end-with-a-single-backslash

            import fileinput
            for i,w in pairs:
                filename = i
                # Btw., you only need to do this once - check if it was already fixed
                if os.path.exists(filename+".bak"):
                    pass
                else:
                    # Note that within the scope of this context manager, print() goes into file.
                    with fileinput.FileInput(filename, inplace=True, backup='.bak') as file:
                        for line in file:
                            print(line.replace(text_to_search, replacement_text), end='')

        print("Visiting script dir",doe2bat_dir,file=logfile)
        os.chdir(doe2bat_dir)
    
        for i,w in pairs:
            simfile = re.sub("\.inp",".sim",i,flags=re.IGNORECASE)
            print(i,"->",simfile,end=": ",flush=True,file=logfile)
            if os.path.exists(os.path.join(target_path,simfile)):
                print("[SIM file already exists - skip]",file=logfile)
            else:
                start = timer()
                output_message = weatherbatch2(w,i,simfiles=target_path,flag="ALL")
                print(output_message.decode(),file=logfile)
                logfile.flush()
                end = timer()
                print("[Saved - elapsed time {:0.1f} s]".format(end-start),file=logfile)
                


if __name__ == '__main__':
    main()